package com.wisedu.coeus.web;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import com.wisedu.coeus.core.bo.LoginUser;
import com.wisedu.coeus.core.exception.CoeusCoreException;
import com.wisedu.coeus.util.Strings;
import com.wisedu.mooc.app.user.bo.UserInfo;

import java.util.UUID;

/**
 * 登录上下文
 */
public final class LoginContext {
    public static final ThreadLocal<LoginUser> LOGIN_HOLDER = new ThreadLocal<LoginUser>();

    public static final ThreadLocal<HttpServletRequest> REQUEST_HOLDER = new ThreadLocal<HttpServletRequest>();

    public static final ThreadLocal<HttpServletResponse> RESPONSE_HOLDER = new ThreadLocal<HttpServletResponse>();

    /**
     * 获取当前的request对象
     * @return
     */
    public static final HttpServletRequest getRequest() {
        return REQUEST_HOLDER.get();
    }

    public static void setRequest(HttpServletRequest request) {
        REQUEST_HOLDER.set(request);
    }

    /**
     * 获取当前response对象
     * @return
     */
    public static final HttpServletResponse getResponse() {
        return RESPONSE_HOLDER.get();
    }

    public static void setResponse(HttpServletResponse response) {
        RESPONSE_HOLDER.set(response);
    }

    /**
     * 获取当前的session对象
     * @return
     */
    private static final HttpSession getHttpSession() {
        HttpServletRequest request = getRequest();
        return request.getSession(true);
    }

    /**
     * 获取当前登录用户
     * @return
     */
    public static LoginUser getLoginUser() {
        LoginUser loginUser = LOGIN_HOLDER.get();
        return loginUser;
    }

    /**
     * 装载当前登录用户
     * @return
     */
    public static LoginUser load() {
        HttpServletRequest request = REQUEST_HOLDER.get();
        LoginUser loginUser = LOGIN_HOLDER.get();
        if(null == loginUser){
            loginUser = loadSession();
        }
        if (null == loginUser) {
        	UserInfo userInfo = new UserInfo();
        	userInfo.setUserId(LoginUser.COEUS_ANONYMOUS_USER_ID);
            loginUser = new LoginUser(userInfo);
            loginUser.setGuest(true);
        }
        setLoginUser(loginUser);
        return loginUser;
    }

    /**
     * 设置登录用户
     * @param loginUser
     */
    public static void setLoginUser(LoginUser loginUser) {
        LOGIN_HOLDER.set(loginUser);
    }

    /**
     * 缓存登录信息
     */
    public static void saveSession() {
        LoginUser loginUser = LOGIN_HOLDER.get();
        HttpSession httpSession = getHttpSession();
        httpSession.setAttribute(LoginUser.COEUS_LOGIN_USER_KEY, loginUser);
    }

    /**
     * 从缓存中装载登录用户
     * @return
     */
    private static LoginUser loadSession() {
        HttpSession httpSession = getHttpSession();
        return (LoginUser) httpSession.getAttribute(LoginUser.COEUS_LOGIN_USER_KEY);
    }

    /**
     * 维护登录上下文中的自定义对象
     * @param key
     * @param value
     * @throws CoeusCoreException
     */
    public static void setAttribute(String key, Object value) throws CoeusCoreException {
        if (getLoginUser() == null) {
            throw new CoeusCoreException(CoeusCoreException.UNINITIALIZED_OBJECT, "Uninitialized object");
        }
        getLoginUser().setAttribute(key, value);
    }

    public static Object getAttribute(String key) throws CoeusCoreException {
        if (getLoginUser() == null) {
            throw new CoeusCoreException(CoeusCoreException.UNINITIALIZED_OBJECT, "Uninitialized object");
        }
        return getLoginUser().getAttribute(key);
    }

    public static void removeAttribute(String key) throws CoeusCoreException {
        if (getLoginUser() == null) {
            throw new CoeusCoreException(CoeusCoreException.UNINITIALIZED_OBJECT, "Uninitialized object");
        }
        getLoginUser().removeAttribute(key);
    }

    /**
     * 清除登录上下文环境
     */
    public static void clear() {
        LOGIN_HOLDER.set(null);
        REQUEST_HOLDER.set(null);
        RESPONSE_HOLDER.set(null);
    }
}
