/**
 * AppConfig.java
 * @author cbPan
 * @date 2012-11-27
 */
package com.wisedu.coeus.web;

import java.util.Locale;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;

import com.wisedu.coeus.helper.ApplicationHelper;
import com.wisedu.coeus.util.Strings;



/**
 * Created 2008-9-1 下午06:34:23 History: 1. 添加返回所有参数Map的方法 2. 添加获取整型、布尔型值的方法
 */
public final class AppConfig implements InitializingBean {

    /** The log. */
    private static Logger log = LoggerFactory.getLogger(AppConfig.class);

    /** The need config. */
    private String[] needConfig;

    /** */
    @SuppressWarnings("unchecked")
    private static Map systemProperties;

    /**
     * Sets the configuration.
     * 
     * @param properties the configuration
     */
    @SuppressWarnings("unchecked")
    public void setConfiguration(final Map properties) {
        systemProperties = properties;
    }

    /**
     * Gets the property.
     * 
     * @param key the key
     * @return the property
     */
    public static String getProperty(final String key) {
    	if(systemProperties==null){
    		systemProperties = (Map) ApplicationHelper.getBean("systemProperties");
    	}
        return (String) systemProperties.get(key);
    }

    /**
     * Gets the trim property.
     * 
     * @param key the key
     * @return the trim property
     */
    public static String getTrimProperty(final String key) {
        return Strings.trimToEmpty((String) systemProperties.get(key));
    }

    /**
     * Gets the property.
     * 
     * @param key the key
     * @param defaultString the default string
     * @return the property
     */
    public static String getProperty(final String key, final String defaultString) {
        String prop = getProperty(key);
        if (null == prop || Strings.isBlank(prop)) {
            return defaultString;
        }
        else {
            return prop;
        }
    }

    /**
     * 获取配置文件中的参数值.
     * 
     * @param key 参数名称.
     * @return 参数值.
     */
    public static int getIntProperty(final String key) {
        return getIntProperty(key, 0);
    }

    /**
     * 获取配置文件中的参数值.
     * 
     * @param key 参数名称.
     * @param defaultValue 默认值.
     * @return 参数值.
     */
    public static int getIntProperty(final String key, int defaultValue) {
        String property = Strings.trimToNull(getProperty(key));
        if (property == null) {
            return defaultValue;
        }
        int value;
        try {
            value = Integer.parseInt(property);
        }
        catch (NumberFormatException e) {
            value = defaultValue;
            log.warn(e.getMessage());
        }
        return value;
    }

    /**
     * 获取配置文件中的参数值.
     * 
     * @param key 参数名称.
     * @param defaultValue 默认值.
     * @return 参数值.
     */
    public static long getLongProperty(final String key, long defaultValue) {
        String property = Strings.trimToNull(getProperty(key));
        if (property == null) {
            return defaultValue;
        }
        long value;
        try {
            value = Long.parseLong(property);
        }
        catch (NumberFormatException e) {
            value = defaultValue;
            log.warn(e.getMessage());
        }
        return value;
    }

    /**
     * 获取配置文件中的整型参数值.
     * 
     * @param key 参数名称.
     * @return 整型参数值.
     */
    public static Integer getIntegerProperty(final String key) {
        return getIntegerProperty(key, null);
    }

    /**
     * 获取配置文件中的整型参数值.
     * 
     * @param key 参数名称.
     * @param defaultValue 缺省值.
     * @return 整型参数值.
     */
    public static Integer getIntegerProperty(final String key, final Integer defaultValue) {
        String strVal = Strings.trimToNull(getProperty(key));
        Integer value = defaultValue;
        if (strVal != null) {
            try {
                value = Integer.parseInt(strVal);
            }
            catch (NumberFormatException e) {
                log.warn(e.getMessage());
            }
        }
        return value;
    }

    /**
     * 获取配置文件中的布尔参数值.
     * 
     * @param key 参数名称.
     * @return 布尔参数值.
     */
    public static boolean getBooleanProperty(final String key) {
        return getBooleanProperty(key, false);
    }

    /**
     * 获取配置文件中的布尔参数.如果文件中没有该参数就返回defaultValue.
     * 
     * @param key 参数名称.
     * @param defaultValue 参数默认值.
     * @return 布尔参数值.
     */
    public static boolean getBooleanProperty(final String key, final boolean defaultValue) {
        if (systemProperties == null) {
            return defaultValue;
        }
        Object value = systemProperties.get(key);
        if (null != value) {
            String strValue = Strings.trimToNull(value.toString());

            return "true".equalsIgnoreCase(strValue) || "on".equalsIgnoreCase(strValue)
                || "yes".equalsIgnoreCase(strValue) || "1".equalsIgnoreCase(strValue);
        }
        else {
            return defaultValue;
        }
    }

    /**
     * Gets the properties.
     * 
     * @return the properties
     */
    @SuppressWarnings("unchecked")
    public static Map getProperties() {
        return systemProperties;
    }

    /**
     * 检查系统必须的配置参数是否设置.
     * 
     * @throws Exception the exception
     */
    public void afterPropertiesSet() throws Exception {
        // 设置应用默认locale
        String localeFlag = Strings.trimToNull((String) systemProperties.get("core.defaultLocale"));
        Locale defaultLoacle = null;
        if (localeFlag != null) {
            defaultLoacle = org.springframework.util.StringUtils.parseLocaleString(localeFlag);
        }
        if (defaultLoacle == null) {
            defaultLoacle = Locale.CHINA;
        }
        Locale.setDefault(defaultLoacle);
        log.info("default locale is '{}'", defaultLoacle);
        // 检查某些关键的配置顶是否存在，不存在就报初始化错误
        final String[] keys = this.needConfig;
        String key;

        for (int i = 0, n = keys.length; i < n; i++) {
            key = Strings.trimToEmpty(keys[i]);

            if (!systemProperties.containsKey(key)) {
                throw new IllegalStateException("Can not find property \"" + key
                    + "\" in core.properties configuration file.");
            }
        }
    }

    /**
     * Sets the need config.
     * 
     * @param needConfig the need config
     */
    public void setNeedConfig(String[] needConfig) {
        this.needConfig = needConfig;
    }
}
